#!/usr/bin/env python
# -*- coding: utf-8 -*-

# ***************************************************************************
# *   Copyright (C) 2013, Paul Lutus                                        *
# *                                                                         *
# *   This program is free software; you can redistribute it and/or modify  *
# *   it under the terms of the GNU General Public License as published by  *
# *   the Free Software Foundation; either version 2 of the License, or     *
# *   (at your option) any later version.                                   *
# *                                                                         *
# *   This program is distributed in the hope that it will be useful,       *
# *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
# *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
# *   GNU General Public License for more details.                          *
# *                                                                         *
# *   You should have received a copy of the GNU General Public License     *
# *   along with this program; if not, write to the                         *
# *   Free Software Foundation, Inc.,                                       *
# *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
# ***************************************************************************

# version date 10-25-2013

VERSION = '1.6'

# no serial support in python 3!

import os, re, signal
import serial
import platform
import webbrowser

import SerialTerminalGUI

import wx,gettext

from wx.lib.embeddedimage import PyEmbeddedImage

class Icon:
  app_icon = PyEmbeddedImage(
    "iVBORw0KGgoAAAANSUhEUgAAACAAAAAgCAMAAABEpIrGAAAAM1BMVEU4AAAEHzciIyEhLWUO"
    "NlRhN7lVVlRnWitKWp5JaYJif5eHkJMpq7d5ongA0tN4zIDU1tOxSUmBAAAAAXRSTlMAQObY"
    "ZgAAAAFiS0dEAIgFHUgAAAAJcEhZcwAADdcAAA3XAUIom3gAAAAHdElNRQfdChkQAw7VZnhR"
    "AAAAxElEQVQ4y52TURKDIAwFBQKSEsT7n7YBtMZBULs/Or4lECdM0z3rgJK7AdlYtXPhEuf0"
    "JkCHQwjh04BSaFNEuBDQe8wQobWmETi2lhXvaeE3NoRQFuavWfGYZmNACQGrUA1eOmul5BlU"
    "yflvVMWoQivEFCP+ckYIwHtiTCk6JTgEzo1BWhZC6ApQBII/K7ChgOhcQApbX3DKhdBhF/Q2"
    "PnJUCroK/Ozh1idDK6kb3N2AMAyZy9rtSe6v19sKpYU9HDbymi+O2Rbu1639FQAAAABJRU5E"
    "rkJggg==")
  

class ConfigManager:

  def __init__(self,parent):
    self.parent = parent
    # config file located at (user home dir)/.programname/classname.ini
    self.program_name = parent.__class__.__name__
    configdir = os.path.join(os.path.expanduser("~"),"." + self.program_name)
    if(not os.path.exists(configdir)): os.makedirs(configdir)
    self.configpath = os.path.join(configdir,self.program_name + ".ini")
    self.logpath = os.path.join(configdir,self.program_name + ".log")

  def read_config(self):
    if(os.path.exists(self.configpath)):
      with open(self.configpath) as f:
        for line in f.readlines():
          name,value = re.search(r'^\s*(.*?)\s*=\s*(.*?)\s*$',line).groups()
          ci = getattr(self.parent.main_frame,name,None)
          if(ci != None):
            dci = dir(ci)
            if("SetValue" in dci):
              ci.SetValue(value == 'True')
            elif("SetStringSelection" in dci):
              ci.SetStringSelection(value)
            elif("SetSize" in dci):
              x,y = map(int,re.sub('.*\((\d+),\s*(\d+)\).*','\\1,\\2',value).split(','))
              ci.SetSize(wx.Size(x,y))
          else:
            print("no object named %s" % name)

  def write_config(self):
    try:
      with open(self.configpath,'w') as f:
        for name in dir(self.parent.main_frame):
          if re.search(r'^k_',name):
            ci = getattr(self.parent.main_frame,name,None)
            if(ci != None):
              dci = dir(ci)
              if("GetValue" in dci):
                f.write("%s = %s\n" % (name,ci.GetValue()))
              elif("GetStringSelection" in dci):
                f.write("%s = %s\n" % (name,ci.GetStringSelection()))
              elif("GetSize" in dci):
                f.write("%s = %s\n" % (name,ci.GetSize()))
    except Exception as e:
      print ("Error: %s" % str(e))

class SerialTerminal:
  
  def write_file(self,path,data):
    with open(path,'w') as f:
      f.write(data)
      
  def read_file(self,path):
    with open(path) as f:
      return f.read() 

  def __init__(self):
    self.config = ConfigManager(self)
    self.title = self.config.program_name + ' ' + VERSION
    gettext.install("app") # replace with the appropriate catalog name
    app = wx.PySimpleApp(0)
    wx.InitAllImageHandlers()
    self.main_frame = SerialTerminalGUI.SerialTerminalGUI(None, wx.ID_ANY, "" )
    self.main_frame.SetTitle(self.title)
    # this allows user frame size choice to be included in the configuration
    self.main_frame.k_frame = self.main_frame
    self.main_frame.SetIcon(Icon.app_icon.GetIcon())
    app.SetTopWindow(self.main_frame)
    self.main_frame.Show()
    self.main_frame.Bind(wx.EVT_CLOSE,self.gui_exit)
    self.operating_system = platform.system()
    self.ansi_state = 0
    self.cursorX = 0
    self.cursorY = 0
    self.stylev = 0
    self.colorv = 0
    self.CHAR_BELL = 7
    self.CHAR_BS = 8
    self.CHAR_LF = 10
    self.CHAR_CR = 13
    self.CHAR_SI = 15
    self.CHAR_TAB = 9
    self.CHAR_ESC = 27
    self.CHAR_DEL = 127
    self.text_colors = (
      -1,
      wx.RED,
      wx.GREEN,
      wx.Color(255,255,0),
      wx.BLUE,
      wx.Color(255,0,255),
      wx.Color(0,255,255),
      wx.WHITE
      )
    self.old_color = False
    self.sys_white_color = wx.WHITE
    self.sys_black_color = wx.BLACK
    self.fg_color = False
    self.bg_color = False
    
    self.loghandle = False
    self.running = False
    self.ser_handle = False
    self.old_satmode = False
    self.old_port = ""
    self.old_rate = ""
    self.ser_port_lock_path = ""
    self.keybuf = ""
    self.comline_history = []
    self.comline_index = 0
    self.comline_current = ""
    self.com_key_event = False
    self.main_key_event = False

    baud_rates = (
      "50",
      "75",
      "110",
      "134",
      "150",
      "200",
      "300",
      "600",
      "1200",
      "1800",
      "2400",
      "4800",
      "9600",
      "19200",
      "38400",
      "57600",
      "115200",
      "230400",
    )

    self.main_frame.k_size_choice.AppendItems([str(x) for x in range(64,0,-1)])
    self.main_frame.k_rate_choice.AppendItems(baud_rates)
    if(re.search('(?i)windows',self.operating_system)):
      devs = ['COM%d' % dev for dev in range(1,9)]
    else:
      devs = [dev for dev in sorted(os.listdir("/dev")) if (re.search('tty(U|S)',dev))]
    self.main_frame.k_port_choice.AppendItems(devs)
    # initial default values
    self.main_frame.k_rate_choice.SetStringSelection('4800')
    self.main_frame.k_port_choice.SetStringSelection('ttyUSB0')
    self.main_frame.k_size_choice.SetStringSelection('10')
    self.main_frame.k_rtscts_checkbox.SetValue(True)
    self.main_frame.k_scroll_checkbox.SetValue(True)
    self.main_frame.k_inverse_checkbox.SetValue(True)
    # now read configuration file if exists
    self.config.read_config()
    self.mainwindow = self.main_frame
    self.mainwindow.SetTitle(self.title)
    #self.mainwindow.SetIcon(gtk.gdk.pixbuf_new_from_xpm_data(Icon.icon))
    self.term_window = self.mainwindow.display_text_ctrl

    # gracefully exit on keyboard and other signals

    signal.signal(signal.SIGTERM, self.gui_exit)
    signal.signal(signal.SIGINT, self.gui_exit)

    # connect controls to actions
    
    self.connect_task()
    self.main_frame.k_log_checkbox.SetToolTipString("Log session content to %s" % (self.config.logpath))
    self.running = True
    # solve the problem that the text display won't accept
    # a color change until until 100ms has passed
    temptimer = wx.Timer(self.main_frame)
    self.main_frame.Bind(wx.EVT_TIMER, self.toggle_control, temptimer,3)
    temptimer.Start(100,oneShot = True)
    self.main_frame.display_text_ctrl.SetFocus()
    self.timer = wx.Timer(self.main_frame)
    self.main_frame.Bind(wx.EVT_TIMER, self.process_io, self.timer,1)
    self.timer.Start(100)
    app.MainLoop()
    
  # associate controls with actions
  def connect_task(self):
    
    self.main_frame.Bind(wx.EVT_CLOSE,self.gui_exit)
    for cb in (
      self.main_frame.k_incoming_cr_checkbox,
      self.main_frame.k_outgoing_cr_checkbox,
      self.main_frame.k_scroll_checkbox,
      self.main_frame.k_echo_checkbox,
      self.main_frame.k_linewrap_checkbox,
      self.main_frame.k_rtscts_checkbox,
      self.main_frame.k_inverse_checkbox,
      self.main_frame.k_log_checkbox,
    ):
      cb.Bind(wx.EVT_CHECKBOX,self.toggle_control)
    for b in (
      (self.main_frame.clear_log_button,self.clear_log),
      (self.main_frame.help_button,self.launch_help),
      (self.main_frame.quit_button,self.gui_exit),
    ):
      b[0].Bind(wx.EVT_BUTTON,b[1])
      
    for ch in (
      self.main_frame.k_size_choice,
      self.main_frame.k_rate_choice,
      self.main_frame.k_port_choice,
    ):
      ch.Bind(wx.EVT_CHOICE,self.toggle_control)
      
    self.main_frame.display_text_ctrl.Bind(wx.EVT_KEY_DOWN, self.keyboard_down)
    self.main_frame.display_text_ctrl.Bind(wx.EVT_KEY_UP, self.keyboard_up)
    self.main_frame.display_text_ctrl.Bind(wx.EVT_CHAR, self.keyboard_char)

  def show_status(self,s):
    self.main_frame.status_label.SetLabel(s)

  def set_term_color(self,normal):
    col = ("gray",self.bg_color)[normal]
    self.term_window.SetBackgroundColour(col)
    
  def launch_help(self,*args):
     webbrowser.open("http://arachnoid.com/SerialTerminal/index.html")

  def gui_exit(self,*args):
    self.running = False
    self.close_serial()
    self.config.write_config()
    self.control_log(False)
    self.main_frame.Destroy()

  def toggle_control(self,*args):
    if(self.running):
      print "toggle control"
      if(self.main_frame.k_inverse_checkbox.GetValue()):
        print "inverse"
        bg = self.sys_white_color
        fg = self.sys_black_color
      else:
        print "normal"
        fg = self.sys_white_color
        bg = self.sys_black_color
      if(not self.fg_color or self.fg_color != fg):
        print "reset %s %s" % (fg.GetAsString(),bg.GetAsString())
        self.set_textctrl_colors(fg,bg)
      self.fg_color = fg
      self.bg_color = bg
      ws = self.main_frame.display_text_ctrl.GetWindowStyle()
      if(self.main_frame.k_linewrap_checkbox.GetValue()):
        ws = ws & ~wx.TE_DONTWRAP | wx.TE_BESTWRAP
      else:
        ws = ws & ~wx.TE_BESTWRAP | wx.TE_DONTWRAP
      self.main_frame.display_text_ctrl.SetWindowStyle(ws)
      self.main_frame.display_text_ctrl.Refresh()
      #self.term_window.set_wrap_mode(wm)
      #self.scroll_to_bottom()
      self.control_log(self.main_frame.k_log_checkbox.GetValue())
      fontsz = int(self.main_frame.k_size_choice.GetStringSelection())
      font = wx.Font(fontsz,wx.TELETYPE, wx.NORMAL, wx.NORMAL, False, u'Monospace')
      self.term_window.SetFont(font)
      self.main_frame.display_text_ctrl.SetFocus()
      self.main_frame.display_text_ctrl.Refresh()
      self.init_serial()

  def set_textctrl_colors(self,fg,bg):
    self.main_frame.display_text_ctrl.SetDefaultStyle(wx.TextAttr(fg,bg))
    if(bg != None):
      print "set default colors %s %s" % (fg.GetAsString(),bg.GetAsString())
      self.main_frame.display_text_ctrl.SetForegroundColour(fg)
      self.main_frame.display_text_ctrl.SetBackgroundColour(bg)
    
  def clear_log(self,*args):
    oldstate = self.main_frame.k_log_checkbox.GetValue()
    self.control_log(False)
    f = open(self.config.logpath,'w')
    f.close()
    self.control_log(oldstate)

  def control_log(self,state):
    self.main_frame.k_log_checkbox.SetValue(state)
    if(state):
      if not (self.loghandle):
        self.loghandle = open(self.config.logpath,'a')
    else:
      if(self.loghandle):
        self.loghandle.close()
        self.loghandle = False

  def write_log(self,s):
    if(self.loghandle):
      self.loghandle.write(s)

  def create_lock_path(self,port):
    return "/var/lock/LCK..%s" % self.main_frame.k_port_choice.GetStringSelection()

  def serial_flag_control(self,state,port=""):
    flagpath = self.create_lock_path(port)
    if(state == 0): # check if exists
      if(self.operating_system == 'Windows'): return False
      return os.path.exists(flagpath)
    elif(state == 1): # create
      if(self.operating_system == 'Windows'): return
      open(flagpath, 'w').close()
      self.ser_port_lock_path = flagpath
    elif(state == 2): # erase
      if(self.operating_system == 'Windows'): return
      if(os.path.exists(self.ser_port_lock_path)): os.remove(self.ser_port_lock_path)
      self.ser_port_lock_path = ""

  def close_serial(self):
    if(self.ser_handle):
      self.ser_handle.close()
      self.ser_handle = False
    self.serial_flag_control(2) # delete old flag

  def init_serial(self,*args):
    if(self.main_frame.k_port_choice.GetStringSelection() != self.old_port \
    or self.main_frame.k_rate_choice.GetStringSelection() != self.old_rate):
      self.set_term_color(True)
      confs = "port %s, rate %s" % (self.main_frame.k_port_choice.GetStringSelection(),self.main_frame.k_rate_choice.GetStringSelection())
      try:
        self.close_serial()
        # test if flag already esists
        if(self.serial_flag_control(0,self.main_frame.k_port_choice.GetStringSelection())):
          raise Exception("Port %s in use" % self.main_frame.k_port_choice.GetStringSelection())
        port = self.main_frame.k_port_choice.GetStringSelection()
        if(self.operating_system == 'Linux'):
          port = "/dev/" + port
        rtsctsv = self.main_frame.k_rtscts_checkbox.GetValue() == 'True'
        self.ser_handle = serial.Serial(port, \
        self.main_frame.k_rate_choice.GetStringSelection(),parity = serial.PARITY_NONE,timeout=0,rtscts=rtsctsv)
        self.show_status("Sucessful serial port intialization: %s" % confs)
        self.serial_flag_control(1,self.main_frame.k_port_choice.GetStringSelection()) # create new flag
      except Exception as e:
        self.close_serial()
        self.show_status("Serial initialization failed: %s. Reason: %s" % (confs,e))
        self.set_term_color(False)
    self.old_port = self.main_frame.k_port_choice.GetStringSelection()
    self.old_rate = self.main_frame.k_rate_choice.GetStringSelection()

  def timed_com(self,a):
    self.keybuf += a.pop(0) + "\n"
    return(len(a) > 0)

  def com_string(self,s):
    array = s.split("|")
    if(len(array) > 1):
      self.timed_com(array)
      self.timer2 = wx.Timer(self.main_frame)
      self.main_frame.Bind(wx.EVT_TIMER, lambda: self.timed_com(array), self.timer2,2)
      self.timer2.Start(500)
      # gobject.timeout_add(500,lambda: self.timed_com(array))
    else:
      self.keybuf += s + "\n"

  def keyboard_up(self,event):
    #print "up",str(event)
    return True
    
  def keyboard_down(self,event):
    #print "down",str(event)
    event.Skip()
    return True
    
  def keyboard_char(self,event):
    k = event.GetKeyCode()
    if(self.ser_handle):
      # print "char",str(event),"[" + unichr(k) + "]",k
      if(k == wx.WXK_UP):
        self.process_outgoing('\033[A',False)
      elif(k == wx.WXK_DOWN):
        self.process_outgoing('\033[B',False)
      elif(k == wx.WXK_RIGHT):
        self.process_outgoing('\033[C',False)
      elif(k == wx.WXK_LEFT):
        self.process_outgoing('\033[D',False)
      elif(k == wx.WXK_BACK):
        self.process_outgoing(unichr(self.CHAR_DEL),False)
      elif(k == self.CHAR_CR):
        self.process_outgoing(unichr(self.CHAR_LF))
        if(self.main_frame.k_outgoing_cr_checkbox.GetValue()):
          self.process_outgoing(unichr(self.CHAR_CR))
      else:
        self.process_outgoing(unichr(k))
        

  def process_outgoing(self,s,echo = True):
    self.ser_handle.write(s)
    if(echo and self.main_frame.k_echo_checkbox.GetValue()):
      self.display_text(s)

  def process_io(self,*args):
    # self.process_keys()
    if(self.ser_handle):
      s = self.ser_handle.read(8192)
      if(len(s) > 0):
        if(not self.main_frame.k_incoming_cr_checkbox.GetValue()):
          s = re.sub("\r","",s)
        s = self.process_ansi(s)
    if(self.loghandle):
      self.loghandle.flush()
    if(self.main_frame.k_scroll_checkbox.GetValue()):
      self.main_frame.display_text_ctrl.Refresh()
    return(True)

  def process_ansi(self,s):
    n = 0
    for c in s:
      if(self.ansi_state == 0):
        if(c == chr(self.CHAR_ESC)):
          self.ansi_state = 1
        else:
          if(c == unichr(self.CHAR_BELL)):
            os.system('beep')
          elif(c == unichr(self.CHAR_BS)):
            # self.ansi_delete_chars(-1)
            self.ansi_adjust_cursor(-1,-1)
          elif(c == unichr(self.CHAR_SI)):
            None # print "CHAR SI detected"
          else:
            self.display_text(c)
      elif(self.ansi_state == 1):
        if(c == '['):
          self.ansi_state = 2
          n = 0
          self.colorv = 0
          self.stylev = 0
      elif(self.ansi_state == 2 or self.ansi_state == 3):
        #print 'final stretch' , c
        if(c >= '0' and c <= '9'):
          n = (n * 10) + (ord(c) - ord('0'))
        elif(c == ';'):
          stylev = n;
          n = 0;
          self.ansi_state = 3
        elif(c == 'H'):
          if (self.ansi_state == 2):
            self.cursorX = n
            self.cursorY = 0
          elif (self.ansi_state == 3):
            self.cursorX = stylev
            self.cursorY = n
          self.ansi_state = 0
        elif(c == 'J'):
          self.ansi_clear(n)
          self.ansi_state = 0
        elif(c == 'K'):
          self.ansi_clear_line(n)
          self.ansi_state = 0
        elif(c == 'm'):
          if(self.ansi_state == 3):
            self.colorv = n % 10
          else:
            self.stylev = n
          # print "color call: %d %d %d" % (n,self.colorv,self.stylev)
          self.set_text_color(self.colorv)
          self.ansi_state = 0
        elif(c == 'C'):
          self.ansi_adjust_cursor(n, 1)
          self.ansi_state = 0
        elif(c == 'D'):
          self.ansi_adjust_cursor(-n, -1)
          self.ansi_state = 0
        elif(c == 'P'):
          self.ansi_delete_chars(n)
          self.ansi_state = 0
        else:
          print("new ansi code in ansi state %d: [%c]" % (self.ansi_state,c))
          self.ansi_state = 0
          
  def set_text_color(self,n):
    # print("set text color: %d" % n)
    col = self.text_colors[n]
    if(col == -1):
      col = self.fg_color
    if(col != self.old_color):
      self.old_color = col
      self.set_textctrl_colors(col,None)
        
  def ansi_clear(self,n):
    print "ansi clear: %d" % n
    n = min(1,n)
  
  def ansi_clear_line(self,n):
    #print "ansi clear line: %d" % n
    if(n == 0): # clear to end of present line
      a = self.main_frame.display_text_ctrl.GetInsertionPoint()
      b = self.main_frame.display_text_ctrl.GetLastPosition()
      self.main_frame.display_text_ctrl.Replace(a,b,"")
    else:
      print "ansi clear line unhandled: %d" % n
    
  def ansi_delete_chars(self,n):
    # print "ansi del chars: %d" % n
    n = min(1,n)
    p = self.main_frame.display_text_ctrl.GetInsertionPoint()
    self.main_frame.display_text_ctrl.Replace(p,p+n,"")
         
  def ansi_adjust_cursor(self,n,defv):
    n = (n,defv)[n == 0]
    # print "adjust_cursor: %d" % n
    p = self.main_frame.display_text_ctrl.GetInsertionPoint()
    self.main_frame.display_text_ctrl.SetInsertionPoint(p + n)
      
  def display_text(self,c):
    if(c != 0):
      p = self.main_frame.display_text_ctrl.GetInsertionPoint()
      self.main_frame.display_text_ctrl.Replace(p,p+1,c)
      self.write_log(c)
      
# end of SerialTerminal class

if __name__ == "__main__":
  SerialTerminal()