/***************************************************************************
 *   Copyright (C) 2004 by Paul Lutus                                      *
 *   lutusp@arachnoid.com                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#include "signalgen.h"
#include "oscillatoraudiodriver.h"
#include "oscillatordisplaydriver.h"
#include <qlineedit.h>
#include <qcombobox.h>
#include <qslider.h>
#include <qlineedit.h>
#include <qmessagebox.h>
#include <qpushbutton.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include "PieGraph.xpm"

const string SignalGen::programName = "SignalGen";
const string SignalGen::programVersion = "1.3";


const char *SignalGen::waveList[] = { "Sine","Triangle","Square","Sawtooth"
                                    };

const char *SignalGen::scopeList[] = { "Disabled","1 Sec","500 ms","200 ms","100 ms","50 ms","20 ms","10 ms"
                                       ,"5 ms","2 ms","1 ms","500 us","200 us","100 us","50 us","10 us"
                                     };

const double SignalGen::scopeRates[] = {
                                           -1.0, 1.0,.5,.2,.1,.05,.02,.01,5e-3,2e-3,1e-3,500e-6,200e-6,100e-6,50e-6,20e-6,10e-6
                                       };

SignalGen::SignalGen(QWidget* parent, const char* name, WFlags fl)
        : MainFrame(parent,name,fl) {

    usrPath = getenv("HOME");
    progPath = usrPath + "/." + programName;
    initHandler = new InitFileHandler(usrPath,programName);
    setIcon(QPixmap(PieGraph));
    setCaption(programName + " " + programVersion);
    isRunning = false;
    enabled = false;
    closing = false;
    osc = new OscillatorAudioDriver(this);
    odd = new OscillatorDisplayDriver(this);
    devicename = "/dev/dsp";
    initialize();
    setDefaults();
    readConfig();
    writeVals();
    readVals();

}

SignalGen::~SignalGen() {
    shutdownOsc(true);
    delete(osc);
    delete(odd);
}

void SignalGen::setDefaults() {
    basefreq = 440.0;
    sampleRate = 44100;
    scopeRange = 8;
    waveform = 0;
    level = .5;
}

void SignalGen::setApplyButtonState(bool s) {
    applyButton->setEnabled(s);
}

void SignalGen::reviseSampleRate(int v) {
    sampleRate = v;
    ostringstream ss;
    ss << v;
    SampleRateText->setText(ss.str());
}

void SignalGen::initialize() {
    WaveformComboBox->insertStrList(waveList,4);
    ScopeComboBox->insertStrList(scopeList,16);
    //writeVals();
}

void SignalGen::writeVals() {
    LevelControl->setValue((int)(level * 100));
    ScopeComboboxChanged(scopeRange);
    ostringstream ss;
    ss << basefreq;
    FrequencyText->setText(ss.str());
    ss.str("");
    ss.clear();
    ss << sampleRate;
    SampleRateText->setText(ss.str());
    DeviceText->setText(devicename.c_str());
    ScopeComboboxChanged(scopeRange);
    WaveformComboBox->setCurrentItem(waveform);
}

void SignalGen::readVals() {
    waveform = WaveformComboBox->currentItem();
    string oldDevice = devicename;
    devicename = (string) (const char*) DeviceText->text();
    istringstream ss(FrequencyText->text());
    ss >> basefreq;
    //FreqSlider->setValue(500);
    //FreqSliderChanged(500);
    FreqSlider->setMaxValue(1000);
    FreqSlider->setValue(500);
    FreqSliderChanged(500);
    ss.clear();
    ss.str(SampleRateText->text());
    int oldSampleRate = sampleRate;
    ss >> sampleRate;
    setApplyButtonState(false);
    newStart(oldDevice != devicename || sampleRate != oldSampleRate);
}

void SignalGen::closeEvent ( QCloseEvent * e ) {
    closing = true;
    shutdownOsc(true);
    writeConfig();
    e->accept();
}

/*$SPECIALIZATION$*/

void SignalGen::LevelChanged(int v) {
    level = (double)v/100.0;
}

//void SignalGen::DeviceChanged() {
//    devicename = (string) (const char*) DeviceText->text();
//    newStart(true);
//}

//void SignalGen::WaveformChanged(int v) {
//    waveform = v;
//}

//void SignalGen::FreqChanged() {
//    istringstream ss(FrequencyText->text());
//    ss >> basefreq;
//    FreqSlider->setValue(500);
//    FreqSliderChanged(500);
//}

void SignalGen::OffCommand() {
    enabled = false;
    shutdownOsc(false);
}

void SignalGen::OnCommand() {
    if(applyButton->isEnabled()) {
        readVals();
    }
    isRunning = true;
    enabled = true;
    newStart(false);
}

void SignalGen::FreqSliderChanged(int v) {
    double q = (1 + (double) v / 500.0);
    double f = basefreq * (q * q) / 4.0;
    bool oldState = isRunning;
    if(oldState) {
        shutdownOsc(false);
    }
    freqn = f;
    char buf[64];
    sprintf(buf,"%.1lf Hz",freqn);
    FreqT->setText(buf);
    if(oldState) {
        newStart(false);
    }
}

void SignalGen::waveformChanged()
{
waveform = WaveformComboBox->currentItem();
}

void SignalGen::ScopeComboboxChanged(int i) {
    ScopeComboBox->setCurrentItem(i);
    scopeRange = i;
    odd->scale = scopeRates[i];
    odd->gd->frame->repaint();
}

//void SignalGen::sampleRateChanged() {
//    istringstream ss(SampleRateText->text());
//    ss >> sampleRate;
//    newStart(true);
//}

void SignalGen::shutdownOsc(bool newDevice) {
    isRunning = false;
    if(osc != NULL && odd != NULL) {
        osc->wait();
        odd->wait();
        odd->gd->frame->repaint();
    }
    if(newDevice) {
        osc->closeDevice();
    }
}

void SignalGen::newStart(bool newDevice) {
    if(enabled && osc != NULL && odd != NULL) {
        shutdownOsc(newDevice);
        isRunning = true;
        if(!osc->running()) {
            if(newDevice) {
                osc->openDevice();
            }
            osc->start(QThread::HighestPriority);
        }
        if(!odd->running()) {
            odd->start(QThread::IdlePriority);
        }
    }
}

void SignalGen::applyPressed() {
    readVals();
}

void SignalGen::enableApply() {
    setApplyButtonState(true);
}

void SignalGen::AboutButtonPressed() {
    QMessageBox::information(this,"About " + programName + " " + programVersion,
                             programName + " " + programVersion + " Copyright 2008, P. Lutus\n"
                             + programName + " is released under the General Public License (GPL).\n\n"
                             + "For more information about " + programName + ", please visit\n"
                             + "http://www.arachnoid.com/" + programName + "."
                            );
}

void SignalGen::readConfig() {
    initHandler->readConfigFile();
    int xv = x();
    int yv = y();
    initHandler->readValue("PosX",xv);
    initHandler->readValue("PosY",yv);
    move(xv,yv);

    xv = width();
    yv = height();
    initHandler->readValue("SizeX",xv);
    initHandler->readValue("SizeY",yv);
    resize(xv,yv);
    initHandler->readValue("AudioDevice",devicename);
    initHandler->readValue("BaseFreq",basefreq);
    initHandler->readValue("SampleRate",sampleRate);
    initHandler->readValue("Waveform",waveform);
    initHandler->readValue("ScopeRange",scopeRange);
    initHandler->readValue("AudioLevel",level);

}

void SignalGen::writeConfig() {
    initHandler->writeValue("PosX",x());
    initHandler->writeValue("PosY",y());
    initHandler->writeValue("SizeX",width());
    initHandler->writeValue("SizeY",height());
    initHandler->writeValue("AudioDevice",devicename);
    initHandler->writeValue("BaseFreq",basefreq);
    initHandler->writeValue("SampleRate",sampleRate);
    initHandler->writeValue("Waveform",waveform);
    initHandler->writeValue("ScopeRange",scopeRange);
    initHandler->writeValue("AudioLevel",level);

    initHandler->writeConfigFile();
}

#include "signalgen.moc"

